/*
 * File Name: MFDESFireAESAuth.c
 *
 * Description: Handles the processing of AES Authentication for MIFARE
 * DESFire EV1 tags
 *
 * Copyright (C) 2014 Texas Instruments Incorporated - http://www.ti.com/
 *
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/

//===============================================================

#include "MFDESFireAESAuth.h"

//===============================================================
//		Global Variables
//===============================================================

extern uint8_t buf[128];

static uint8_t IV[16] = {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};

//===============================================================

//===================================================================
// NAME: uint8_t Aes_authenticate(uint8_t * pui8SessionKey, uint8_t * pui8RndA, uint8_t * pui8Key)
//
// BRIEF:	Used for AES Authentication of MFDESFire EV1 card.
//
// INPUTS:
//	Parameters:
//		uint8_t		*pui8SessionKey		Pointer to array to storage the Session Key
//		uint8_t		*pui8RndA			Pointer to array that has the RndA values
//		uint8_t		*pui8Key			Pointer to array containing the AES key
//
// OUTPUTS:
//	Parameters:
//		uint8_t	ui8Status		Returns the following:
//								"STATUS_FAIL" => Error has occurred
// 								"STATUS_SUCCESS" => Layer 4 successful
//
//===================================================================

uint8_t Aes_authenticate(uint8_t * pui8SessionKey, uint8_t * pui8RndA, uint8_t * pui8Key)
{
	uint8_t pui8EncryptionBuf[32];
	uint8_t pui8TempKey1[16];
	uint8_t pui8TempKey2[16];
	uint8_t ui8Index;
	uint8_t ui8BufLength = 0;
	uint8_t ui8Status = STATUS_FAIL;

	///Authentication Start -- 3 phases

	buf[ui8BufLength++] = 0x8F;
	buf[ui8BufLength++] = 0x91;
	buf[ui8BufLength++] = 0x3D;
	buf[ui8BufLength++] = 0x00;
	buf[ui8BufLength++] = 0x40;

	buf[ui8BufLength++] = 0x0A; 		// I-block (Layer 4)
	buf[ui8BufLength++] = CID;			// CID
	buf[ui8BufLength++] = AesCommand(); // AES Auth
	buf[ui8BufLength++] = 0x00; 		// KeyNo

	Trf797xRawWrite(&buf[0], ui8BufLength);
	ui8BufLength = 0;

	Trf797xSetRxTxState(1);

	Trf797xIrqWaitTimeout(5,30);		// 5 millisecond TX timeout, 30 millisecond RX timeout

	if (buf[3] != StatusCode1())
	{
		// Error occurred
		return STATUS_FAIL;
	}

	//Encryption start for Authentication phase 2

	// AES code destroys key -- copying for code to work until AES changed.
	for (ui8Index=0;ui8Index<16;ui8Index++)
	{
		pui8TempKey1[ui8Index] = pui8Key[ui8Index];
	}

	for (ui8Index=0;ui8Index<16;ui8Index++)
	{
		pui8EncryptionBuf[ui8Index] = buf[4+ui8Index];	//EncBuf == Capture ek(RndB) for Decryption
	}
	for (ui8Index=0;ui8Index<16;ui8Index++)
	{
		IV[ui8Index] = pui8EncryptionBuf[ui8Index];		// Set IV
	}

	aes_enc_dec(pui8EncryptionBuf, pui8TempKey1, 1);

	// Now we have RanB, so we can already save the Session Key

	// SessionKey == RndA[0..3],RndB[0..3],RndA[12..15],RndB[12..15]
	pui8SessionKey[0] = pui8RndA[0];				// RndA[0..3]
	pui8SessionKey[1] = pui8RndA[1];
	pui8SessionKey[2] = pui8RndA[2];
	pui8SessionKey[3] = pui8RndA[3];

	pui8SessionKey[4] = pui8EncryptionBuf[0];		// RndB[0..3]
	pui8SessionKey[5] = pui8EncryptionBuf[1];
	pui8SessionKey[6] = pui8EncryptionBuf[2];
	pui8SessionKey[7] = pui8EncryptionBuf[3];

	pui8SessionKey[8] =  pui8RndA[12];				// RndA[12..15]
	pui8SessionKey[9] =  pui8RndA[13];
	pui8SessionKey[10] = pui8RndA[14];
	pui8SessionKey[11] = pui8RndA[15];

	pui8SessionKey[12] = pui8EncryptionBuf[12];		// RndB[12..15]
	pui8SessionKey[13] = pui8EncryptionBuf[13];
	pui8SessionKey[14] = pui8EncryptionBuf[14];
	pui8SessionKey[15] = pui8EncryptionBuf[15];

	//Decrypted Random# B rotate left by 8bits, fill right half of EncBuf
	pui8EncryptionBuf[16] = pui8EncryptionBuf[1];
	pui8EncryptionBuf[17] = pui8EncryptionBuf[2];
	pui8EncryptionBuf[18] = pui8EncryptionBuf[3];
	pui8EncryptionBuf[19] = pui8EncryptionBuf[4];
	pui8EncryptionBuf[20] = pui8EncryptionBuf[5];
	pui8EncryptionBuf[21] = pui8EncryptionBuf[6];
	pui8EncryptionBuf[22] = pui8EncryptionBuf[7];
	pui8EncryptionBuf[23] = pui8EncryptionBuf[8];
	pui8EncryptionBuf[24] = pui8EncryptionBuf[9];
	pui8EncryptionBuf[25] = pui8EncryptionBuf[10];
	pui8EncryptionBuf[26] = pui8EncryptionBuf[11];
	pui8EncryptionBuf[27] = pui8EncryptionBuf[12];
	pui8EncryptionBuf[28] = pui8EncryptionBuf[13];
	pui8EncryptionBuf[29] = pui8EncryptionBuf[14];
	pui8EncryptionBuf[30] = pui8EncryptionBuf[15];
	pui8EncryptionBuf[31] = pui8EncryptionBuf[0];

	//Put Random# A in front half of EncBuf
	for (ui8Index=0;ui8Index<16;ui8Index++)
	{
		pui8EncryptionBuf[ui8Index] = pui8RndA[ui8Index];
	}

	for(ui8Index=0;ui8Index<16;ui8Index++)
	{
		pui8EncryptionBuf[ui8Index] ^= IV[ui8Index];
	}

	// AES code destroys key, must copy to a temp buffer
	for (ui8Index=0;ui8Index<16;ui8Index++)
	{
		pui8TempKey2[ui8Index] = pui8Key[ui8Index];
	}

	aes_enc_dec(pui8EncryptionBuf, pui8TempKey2, 0);

	for(ui8Index=0;ui8Index<16;ui8Index++)
	{
		pui8EncryptionBuf[ui8Index+16] ^= pui8EncryptionBuf[ui8Index];
	}

	aes_enc_dec(&pui8EncryptionBuf[16], pui8TempKey1, 0);

	for(ui8Index=0;ui8Index<16;ui8Index++)
	{
		IV[ui8Index] = pui8EncryptionBuf[16+ui8Index];
	}

	buf[ui8BufLength++] = 0x8F;
	buf[ui8BufLength++] = 0x91;
	buf[ui8BufLength++] = 0x3D;
	buf[ui8BufLength++] = 0x02;
	buf[ui8BufLength++] = 0x30;

	buf[ui8BufLength++] = 0x0B; 	// I-block (Layer 4)
	buf[ui8BufLength++] = CID; 		// CID
	buf[ui8BufLength++] = StatusCode1();

	for (ui8Index=0; ui8Index<32; ui8Index++)
	{
		buf[ui8BufLength++] = pui8EncryptionBuf[ui8Index];
	}

	Trf797xRawWrite(&buf[0], ui8BufLength);
	ui8BufLength = 0;

	Trf797xSetRxTxState(1);

	Trf797xIrqWaitTimeout(15,40);	// 15 millisecond TX timeout, 40 millisecond RX timeout

	if (buf[3] != StatusCode2())
	{
		// Error occurred
		return STATUS_FAIL;
	}

	for (ui8Index=0;ui8Index<16;ui8Index++)
	{
		pui8EncryptionBuf[ui8Index] = buf[4+ui8Index];	// EncBuf == Capture ek(RndB) for Decryption
	}

	// AES code destroys key, must copy to a temp buffer
	for (ui8Index=0;ui8Index<16;ui8Index++)
	{
		pui8TempKey2[ui8Index] = pui8Key[ui8Index];
	}
	aes_enc_dec(pui8EncryptionBuf, pui8TempKey2, 1);

	for(ui8Index=0; ui8Index<16; ui8Index++)
	{  //Testing if receiving correct RndA'
		pui8EncryptionBuf[ui8Index] ^= IV[ui8Index];
	}

	if(pui8EncryptionBuf[0] == pui8SessionKey[1])
	{ //Indices are rotated due to the fact that we are comparing Rnd' with RndA
		LED_15693_ON;
		ui8Status = STATUS_SUCCESS;
	}

	//Authentication completed. Zero out IV[]
	for(ui8Index=0; ui8Index<16; ui8Index++)
	{
		IV[ui8Index] = 0;
	}

	McuDelayMillisecond(10);

	return ui8Status;
}
